using System;
using System.Runtime.InteropServices;

namespace Sony
{
	namespace NP
	{
		public class Messaging
		{
			// Structure for retrieving a message attachement (internal only).
			[StructLayout(LayoutKind.Sequential, CharSet = CharSet.Ansi, Pack = 0)]
			struct MessageAttachment
			{
				public int dataSize;		// Size of the data in bytes.
				IntPtr _data;
				public byte[] data			// Byte array containing the data.
				{
					get {
						byte[] bytes = new byte[dataSize];
						Marshal.Copy(_data, bytes, 0, dataSize);
						return bytes;
					}
				}
			}

			// Structure for retrieving an in-game data message (internal only).
			[StructLayout(LayoutKind.Sequential, CharSet = CharSet.Ansi, Pack = 0)]
			struct InGameDataMessageInternal
			{
				IntPtr _npID;			// NpID, contains data corresponding to the SceNpID structure, this is the messages sender.
				int npIDSize;			// Size of the npID byte array.
				public int messageID;
				public int dataSize;
				IntPtr _data;

				public byte[] fromNpID
				{
					get
					{
						byte[] bytes = new byte[npIDSize];
						Marshal.Copy(_npID, bytes, 0, npIDSize);
						return bytes;
					}
				}
				public byte[] data			// Byte array containing the data.
				{
					get {
						byte[] bytes = new byte[dataSize];
						Marshal.Copy(_data, bytes, 0, dataSize);
						return bytes;
					}
				}
			};

			// Structure for retrieving an in-game data message.
			public struct InGameDataMessage
			{
				public int messageID;		// The messages id.
				public byte[] fromNpID;		// NpID of the sender.
				public byte[] data;			// Byte array containing the data.
			};

			// Class for passing message params to the plugin.
			[StructLayout(LayoutKind.Sequential, CharSet = CharSet.Ansi, Pack = 0)]
			public class MsgRequest
			{
				IntPtr _body;						// The body text of the message.
				IntPtr _data;						// Attached data.
				int dataSize;						// Size of the attached data in bytes.
				public int expireMinutes;			// Message expire time in minutes.

				// PS4 only.
				public int npIDCount;				// Number npIDs the user can select in the message dialog.
				IntPtr _dataName;					// The data name.
				IntPtr _dataDescription;			// The data description.
				IntPtr _iconPath;					// The path to an icon, which needs to be displayed in the message.

				public string body { set { _body = Marshal.StringToCoTaskMemAnsi(value); } }
				public byte[] data
				{
					set
					{
						dataSize = value.Length;
						_data = Marshal.AllocCoTaskMem(value.Length);
						Marshal.Copy(value, 0, _data, value.Length);
					}
				}
				public string dataName { set { _dataName = Marshal.StringToCoTaskMemAnsi(value); } }
				public string dataDescription { set { _dataDescription = Marshal.StringToCoTaskMemAnsi(value); } }
				public string iconPath { set { _iconPath = Marshal.StringToCoTaskMemAnsi(value); } }

				~MsgRequest()
				{
					Marshal.FreeCoTaskMem(_body);
					Marshal.FreeCoTaskMem(_data);
					Marshal.FreeCoTaskMem(_dataName);
					Marshal.FreeCoTaskMem(_dataDescription);
					Marshal.FreeCoTaskMem(_iconPath);
				}
			};

			[DllImport("UnityNpToolkit")]
			private static extern bool PrxMessagingIsBusy();
			[DllImport("UnityNpToolkit", CharSet = CharSet.Ansi)]
			private static extern bool PrxMessagingSendInGameDataMessage(byte[] npID, byte[] data, int dataSize);
			[DllImport("UnityNpToolkit", CharSet = CharSet.Ansi)]
			private static extern bool PrxMessagingSendMessage(MsgRequest msgRequest);	//string text, int expireMinutes, byte[] data, int dataSize);
			[DllImport("UnityNpToolkit", CharSet = CharSet.Ansi)]
			private static extern bool PrxMessagingSendGameInvite(MsgRequest msgRequest);	//string text, int expireMinutes, byte[] data, int dataSize);
			[DllImport("UnityNpToolkit")]
			private static extern bool PrxMessagingShowMessageDialog();
			[DllImport("UnityNpToolkit")]
			private static extern bool PrxMessagingShowInviteDialog();
			[DllImport("UnityNpToolkit")]
			private static extern bool PrxMessagingGetMessageAttachment(out MessageAttachment attachment);
			[DllImport("UnityNpToolkit")]
			private static extern bool PrxMessagingGetGameInviteAttachment(out MessageAttachment attachment);
			[DllImport("UnityNpToolkit")]
			private static extern bool PrxHasInGameDataMessage();
			[DllImport("UnityNpToolkit")]
			private static extern bool PrxGetFirstInGameDataMessage(out InGameDataMessageInternal message);
			[DllImport("UnityNpToolkit")]
			private static extern bool PrxRemoveFirstInGameDataMessage();

			// Is the messaging service busy?
			public static bool IsBusy()
			{
				return PrxMessagingIsBusy();
			}

			// Send an in-game data message.
			// npID, specifies the recipient for the message, they must be in the same game.
			// data, byte array containing the data to send, a maximumu of 512 bytes.
			public static bool SendInGameDataMessage(byte[] npID, byte[] data)
			{
				return PrxMessagingSendInGameDataMessage(npID, data, data.Length);
			}
			
			// Send a message with an attachement, if no attachment required pass data=null.
			public static bool SendMessage(MsgRequest request)
			{
				return PrxMessagingSendMessage(request);	//text, expireMinutes, data, data.Length);
			}
			
			// Send a game invite with an attachement, if no attachment required pass data=null.
			public static bool SendGameInvite(MsgRequest request)
			{
				return PrxMessagingSendGameInvite(request);	//text, expireMinutes, data, data.Length);
			}

			// Show the system message dialog.
			public static bool ShowRecievedDataMessageDialog()
			{
				return PrxMessagingShowMessageDialog();
			}

			// Show the system invite message dialog.
			public static bool ShowRecievedInviteDialog()
			{
				return PrxMessagingShowInviteDialog();
			}

			// Get the attachment for the last retrieved game invite.
			public static byte[] GetGameInviteAttachment()
			{
				MessageAttachment attachement = new MessageAttachment();
				PrxMessagingGetGameInviteAttachment(out attachement);
				return attachement.data;
			}
			
			// Get the attachment for the last retrieved message.
			public static byte[] GetMessageAttachment()
			{
				MessageAttachment attachement = new MessageAttachment();
				PrxMessagingGetMessageAttachment(out attachement);
				return attachement.data;
			}

			// Has in-game data been recieved?
			public static bool InGameDataMessagesRecieved()
			{
				return PrxHasInGameDataMessage();
			}

			// Get an in-game data message. Typically called in a loop, i.e.
			// while(Sony.NP.Messaging.InGameDataMessagesRecieved())
			// {
			//	Sony.NP.Messaging.InGameDataMessage igm = Sony.NP.Messaging.GetInGameDataMessage();
			//  ...do something with the messaeg...
			//}
			// NOTE that this consumes the messages from the message cache, once you have retrieved a message it can not be retrieved again.
			public static InGameDataMessage GetInGameDataMessage()
			{
				InGameDataMessageInternal igmi = new InGameDataMessageInternal();
				PrxGetFirstInGameDataMessage(out igmi);
				InGameDataMessage igm = new InGameDataMessage();
				igm.fromNpID = igmi.fromNpID;
				igm.messageID = igmi.messageID;
				igm.data = igmi.data;
				PrxRemoveFirstInGameDataMessage();
				return igm;
			}
			
			// Messaging event delegates.
			public static event Messages.EventHandler OnMessageSent;
			public static event Messages.EventHandler OnMessageNotSent;
			public static event Messages.EventHandler OnMessageCanceled;
			public static event Messages.EventHandler OnSessionInviteMessageRetrieved;
			public static event Messages.EventHandler OnCustomInviteMessageRetrieved;
			public static event Messages.EventHandler OnCustomDataMessageRetrieved;
			public static event Messages.EventHandler OnInGameDataMessageRetrieved;
			public static event Messages.EventHandler OnMessageNotSentFreqTooHigh;
			public static event Messages.EventHandler OnMessageSessionInviteReceived;
			public static event Messages.EventHandler OnMessageSessionInviteAccepted;

			// Message processing.
			public static bool ProcessMessage(Messages.PluginMessage msg)
			{
				switch (msg.type)
				{
					case NP.Messages.MessageType.kNPToolKit_MessagingSent:
						if (OnMessageSent != null) OnMessageSent(msg);
						break;
						
					case NP.Messages.MessageType.kNPToolKit_MessagingNotSent:
						if (OnMessageNotSent != null) OnMessageNotSent(msg);
						break;

					case NP.Messages.MessageType.kNPToolKit_MessagingNotSentFreqTooHigh:
						if (OnMessageNotSentFreqTooHigh != null) OnMessageNotSentFreqTooHigh(msg);
						break;
						
					case NP.Messages.MessageType.kNPToolKit_MessagingCanceled:
						if (OnMessageCanceled != null) OnMessageCanceled(msg);
						break;
						
					case NP.Messages.MessageType.kNPToolKit_MessagingSessionInviteRetrieved:
						if (OnSessionInviteMessageRetrieved != null) OnSessionInviteMessageRetrieved(msg);
						break;
						
					case NP.Messages.MessageType.kNPToolKit_MessagingCustomInviteRetrieved:
						if (OnCustomInviteMessageRetrieved != null) OnCustomInviteMessageRetrieved(msg);
						break;
						
					case NP.Messages.MessageType.kNPToolKit_MessagingDataMessageRetrieved:
						if (OnCustomDataMessageRetrieved != null) OnCustomDataMessageRetrieved(msg);
						break;

					case NP.Messages.MessageType.kNPToolKit_MessagingInGameDataMessageRetrieved:
						if (OnInGameDataMessageRetrieved != null) OnInGameDataMessageRetrieved(msg);
						break;
          					
                   case NP.Messages.MessageType.kNPToolKit_MessagingSessionInviteReceived:
						if (OnMessageSessionInviteReceived != null) OnMessageSessionInviteReceived(msg);
						break;
                    					
                    case NP.Messages.MessageType.kNPToolKit_MessagingSessionInviteAccepted:
						if (OnMessageSessionInviteAccepted != null) OnMessageSessionInviteAccepted(msg);
						break;
				}
				
				return false;
			}
			
		}
	}
}
